﻿[CmdletBinding()]
    param(
        [nullable[bool]] $IsSPOnline
    )

#NOTE: Please go to skybow site https://my.skybow.com/hc/en-us/sections/360003398160-Release-Notes-Downloads and download the latest version of skybow modern forms addin and put into local spfx folder

. .\utils.ps1


$LocalPath = ".\export"

if( $null -eq $IsSPOnline )
{
    $IsSPOnline = Prompt-GetIsSharePointOnline
}
if( $IsSPOnline ){
    $LocalPathSPFxAddin =  ".\spfx\skybow-modern-forms.sppkg"
}
else{
    $LocalPathSPFxAddin =  ".\spfx\skybow-modern-forms-2019.sppkg"
}

$ModernFormsCustomActionComponentId = 'd110e6ba-3612-4fd4-8b34-231a7f771d18'
$ConfigurationStorageLib = 'mFormsStoreLibrary'
$ConfigurationStorageFolder = $ConfigurationStorageLib + '/Configuration/Customization/'

Load-Prerequisition -isSPOnline $IsSPOnline
Write-Host "-------------------------------------------------------------------" -ForegroundColor White
$web = ConnectToSPOnline -message 'Enter site URL where you want to import Modern Forms customization' -isSPOnline $IsSPOnline

function Ensure-ModernForms-Resources(){
    $lib = Get-PnPList $ConfigurationStorageLib
    if( -not $lib ){
        $lib = New-PnPList -Title $ConfigurationStorageLib -Template DocumentLibrary
        Set-PnPList -Identity $ConfigurationStorageLib -Hidden $true
        Write-Host Created $lib.Title configuration storage library -ForegroundColor Green            
    }

    $action = Get-ModernForms-CustomAction
    if(!$action.Name){
        $props = @{Forms = @(); Enabled = $true; SolutionId = ''}
        Add-PnPCustomAction -Name 'skyForms_' -ClientSideComponentId $ModernFormsCustomActionComponentId -ClientSideComponentProperties (ConvertTo-Json $props) -Title 'Modern Forms' -Location 'ClientSideExtension.ApplicationCustomizer'
        Write-Host Registered modern forms app customizer action -ForegroundColor Green    
    }
}

function Import-ListCustomization($ListName){
    $LocalFolderPath = $LocalPath + '\' + $ListName;
    $LocalFormsFolderPath = $LocalFolderPath + "\Forms";
    $LocalListConfigurationFilePath = $LocalFolderPath+"\configuration.json"

    $list = Get-PnPList | Where-Object {$_.RootFolder.ServerRelativeUrl -like "*/$ListName"}
    if(!$list.Title){
        Write-Host $ListName - List not exist! -ForegroundColor Red
    }
    else{
        Write-Host Importing $list.Title `($listName`) list customization...
        
        $props = Get-ModernForms-CustomAction-Props
        
        $strListConf = Get-Content $LocalListConfigurationFilePath | Out-String
        $strListConf = UnTokenizeString -content $strListConf -list $list
        $jsonListConf = ConvertFrom-Json $strListConf
        if(!$jsonListConf -or !$jsonListConf.Forms){
            Write-Host List `'$listName`' configuration.json file is broken and cannot be imported. -ForegroundColor Red
        }
        else{
            $dirty = $false
            foreach($listForm in $jsonListConf.Forms){
                if($listForm.Enabled){
                    $props.Forms = $props.Forms | Where-object { $_.FormUrl -ne $listForm.FormUrl }
                    $props.Forms = [Array]$props.Forms + $listForm
                    $dirty = $true;
                }
            }
            if($dirty){
                Update-ModernForms-CustomAction-Props -props $props
            }
        }        
        
        $ListFormJSONFileNames = Get-ChildItem -Path $LocalFormsFolderPath | ?{ -not $_.PSIsContainer } | Where-Object { $_.Name -like '*.aspx.json' } | Select-Object -Property "Name" -ExpandProperty "Name"
        foreach( $formJSONFileName in $ListFormJSONFileNames ){
            $filePath = $LocalFormsFolderPath + '\' + $formJSONFileName;
            $file = Add-PnPFile -Path $filePath -Folder $ConfigurationStorageFolder -NewFileName ($list.Id.ToString() + '_'+ $formJSONFileName) -ErrorAction SilentlyContinue            
        }

        Write-Host Imported $list.Title list customization -ForegroundColor Green
    }
}


function UploadSPFxToAppCatalog(){
    if( 0 -eq (Get-PnPApp | Where-Object {$_.Title -eq 'skybow Modern Forms'}).Length ){
        Add-PnPApp -Path $LocalPathSPFxAddin -Publish
        Write-Host Added 'skybow Modern Forms' add-in to tenant app catalog -ForegroundColor Green
    }
    else{
        Write-Host skybow Modern Forms already exists in app catalog
    }
}

function AddSPFxToSite(){
    if( 0 -eq (Get-PnPAppInstance | Where-Object {$_.Title -eq 'skybow Modern Forms'}).Length ){
        $app = (Get-PnPApp | Where-Object {$_.Title -eq 'skybow Modern Forms'})[0]
        if( $app ){
            Write-Host Deploying 'skybow Modern Forms' add-in to site...
            Install-PnPApp -Identity $app.Id
            Write-Host Deployed 'skybow Modern Forms' add-in to site -ForegroundColor Green
        }
        else{
            Write-Host skybow Modern Forms add-in is already added to site
        }
    }
    else{
        Write-Host Cannot find skybow Modern Forms add-in in app catalog -ForegroundColor Red
    }
}

if($web){
    $ListNamesToImport = Get-ChildItem -Path $LocalPath | ?{ $_.PSIsContainer } | Select-Object -Property "Name" -ExpandProperty "Name"

    $choices = New-Object Collections.ObjectModel.Collection[Management.Automation.Host.ChoiceDescription]
    $choices.Add((New-Object Management.Automation.Host.ChoiceDescription -ArgumentList '&Yes'))
    $choices.Add((New-Object Management.Automation.Host.ChoiceDescription -ArgumentList '&No'))
    $decision = $Host.UI.PromptForChoice('Confirmation', ($ListNamesToImport -join ', ') + ' list(s) customization will be imported. Continue?', $choices, 0)
    if ($decision -eq 0) {
        Write-Host 'Starting...'
        Ensure-ModernForms-Resources
        foreach( $listName in $ListNamesToImport){
            Import-ListCustomization -ListName $listName
        }

        $choices = New-Object Collections.ObjectModel.Collection[Management.Automation.Host.ChoiceDescription]
        $choices.Add((New-Object Management.Automation.Host.ChoiceDescription -ArgumentList '&Yes'))
        $choices.Add((New-Object Management.Automation.Host.ChoiceDescription -ArgumentList '&No'))
        $decision = $Host.UI.PromptForChoice('Confirmation', 'Upload and deploy skybow Modern Forms add-in from '+$LocalPathSPFxAddin+'?', $choices, 0)
        if( $decision -eq 0 ){
            Write-Host 'Installing skybow Modern Forms add-in...'
            UploadSPFxToAppCatalog
            AddSPFxToSite
        }
    }
}

Read-Host -Prompt "Press Enter to quit..."